from typing import Annotated

from fastapi import APIRouter, Depends, Query, Request, Response

from felix_io_api.api import Tags, get_shared
from .constants import TEXT_JAVASCRIPT


#
# Javascript
#
class JavaScriptResponse(Response):
    media_type = TEXT_JAVASCRIPT


js_router = APIRouter()


@js_router.get("/js/device/{device_id}/bitfield/{name}", response_class=JavaScriptResponse, tags=[Tags.javascript])
async def js_device_bitfield_name(shared: Annotated[dict, Depends(get_shared)],
                                  request: Request,
                                  device_id:  int,
                                  name: str
                                  ) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="bitfield.js", media_type=TEXT_JAVASCRIPT, context={"device_id": device_id,
                                                                                  "name": name,
                                                                                  "ajax_url": f"/data/device/{device_id}/bitfield/{name}"}
    )


@js_router.get("/js/device/{device_id}/bitfield", response_class=JavaScriptResponse, tags=[Tags.javascript])
async def js_device_bitfield(shared: Annotated[dict, Depends(get_shared)],
                             request: Request,
                             device_id:  int,
                             name: Annotated[list[str] | None, Query()] = None
                             ) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="bitfield.js", media_type=TEXT_JAVASCRIPT, context={"simulate": shared.simulate,
                                                                                  "hostname": shared.hostname,
                                                                                  "device_id": device_id,
                                                                                  "ajax_url": f"/data/device/{device_id}/bitfield",
                                                                                  "ajax_data": name}
    )


@js_router.get("/js/device/{device_id}/i2c/{name}", response_class=JavaScriptResponse, tags=[Tags.javascript])
async def js_device_i2c_name(shared: Annotated[dict, Depends(get_shared)],
                             request: Request,
                             device_id:  int,
                             name: str
                             ) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="bitfield.js", media_type=TEXT_JAVASCRIPT, context={"device_id": device_id,
                                                                                  "name": name,
                                                                                  "ajax_url": f"/data/device/{device_id}/i2c/{name}"}
    )


@js_router.get("/js/device/{device_id}/i2c", response_class=JavaScriptResponse, tags=[Tags.javascript])
async def js_device_i2c(shared: Annotated[dict, Depends(get_shared)],
                        request: Request,
                        device_id:  int,
                        name: Annotated[list[str] | None, Query()] = None
                        ) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="bitfield.js", media_type=TEXT_JAVASCRIPT, context={"simulate": shared.simulate,
                                                                                  "hostname": shared.hostname,
                                                                                  "device_id": device_id,
                                                                                  "ajax_url": f"/data/device/{device_id}/i2c",
                                                                                  "ajax_data": name}
    )


@js_router.get("/js/device/{device_id}/alias/{name}", response_class=JavaScriptResponse, tags=[Tags.javascript])
async def js_device_alias_name(shared: Annotated[dict, Depends(get_shared)],
                               request: Request,
                               device_id:  int,
                               name: str
                               ) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="bitfield.js", media_type=TEXT_JAVASCRIPT, context={"device_id": device_id,
                                                                                  "name": name,
                                                                                  "ajax_url": f"/data/device/{device_id}/alias/{name}"}
    )


@js_router.get("/js/device/{device_id}/alias", response_class=JavaScriptResponse, tags=[Tags.javascript])
async def js_device_alias(shared: Annotated[dict, Depends(get_shared)],
                          request: Request,
                          device_id:  int,
                          name: Annotated[list[str] | None, Query()] = None
                          ) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="bitfield.js", media_type=TEXT_JAVASCRIPT, context={"simulate": shared.simulate,
                                                                                  "hostname": shared.hostname,
                                                                                  "device_id": device_id,
                                                                                  "ajax_url": f"/data/device/{device_id}/alias",
                                                                                  "ajax_data": name}
    )


@js_router.get("/js/device/{device_id}/group/{name}", response_class=JavaScriptResponse, tags=[Tags.javascript])
async def js_device_group_name(shared: Annotated[dict, Depends(get_shared)],
                               request: Request,
                               device_id:  int,
                               name: str
                               ) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="bitfield.js", media_type=TEXT_JAVASCRIPT, context={"device_id": device_id,
                                                                                  "name": name,
                                                                                  "ajax_url": f"/data/device/{device_id}/group/{name}"}
    )


@js_router.get("/js/device/{device_id}/group", response_class=JavaScriptResponse, tags=[Tags.javascript])
async def js_device_group(shared: Annotated[dict, Depends(get_shared)],
                          request: Request,
                          device_id:  int,
                          name: Annotated[list[str] | None, Query()] = None
                          ) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="bitfield.js", media_type=TEXT_JAVASCRIPT, context={"simulate": shared.simulate,
                                                                                  "hostname": shared.hostname,
                                                                                  "device_id": device_id,
                                                                                  "ajax_url": f"/data/device/{device_id}/group",
                                                                                  "ajax_data": name}
    )


@js_router.get("/js/device/{device_id}/{name}", response_class=JavaScriptResponse, tags=[Tags.javascript])
async def js_device_name(shared: Annotated[dict, Depends(get_shared)],
                         request: Request,
                         device_id:  int,
                         name: str
                         ) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="bitfield.js", media_type=TEXT_JAVASCRIPT, context={"device_id": device_id,
                                                                                  "name": name,
                                                                                  "ajax_url": f"/data/device/{device_id}/{name}"}
    )


@js_router.get("/js/device/{device_id}", response_class=JavaScriptResponse, tags=[Tags.javascript])
async def js_device(shared: Annotated[dict, Depends(get_shared)],
                    request: Request,
                    device_id:  int,
                    name: Annotated[list[str] | None, Query()] = None
                    ) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="bitfield.js", media_type=TEXT_JAVASCRIPT, context={"simulate": shared.simulate,
                                                                                  "hostname": shared.hostname,
                                                                                  "device_id": device_id,
                                                                                  "ajax_url": f"/data/device/{device_id}",
                                                                                  "ajax_data": name}
    )


@js_router.get("/js/device_list", response_class=JavaScriptResponse, tags=[Tags.javascript])
async def js_device_list(shared: Annotated[dict, Depends(get_shared)],
                         request: Request
                         ) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="device_list.js", media_type=TEXT_JAVASCRIPT
    )
