import asyncio
import socket

from contextlib import asynccontextmanager
# from typing import Annotated

from fastapi import FastAPI, Depends
from fastapi.staticfiles import StaticFiles
from fastapi.templating import Jinja2Templates

from redislite import Redis

from apscheduler.schedulers.asyncio import AsyncIOScheduler

from .config import get_settings
from .driver import Driver
from .util import Util

from felix_io_api.api import api_router, title, summary, version, description, get_shared
from .support import support_router
from .list import list_router
from .js import js_router
from .data import data_router


async def clear_latched_registers(device_id: int, bitfield_name: str):
    # logger = Util.get_logger(__name__, get_settings().log_level)
    # logger.debug(f"Clearing {device_id}:{bitfield_name}")
    get_shared().driver.device(device_id).bitfield(bitfield_name).set(1)


@asynccontextmanager
async def lifespan(app: FastAPI):
    # Start
    logger = Util.get_logger(__name__, get_settings().log_level)
    # import logging
    # # logger = logging.getLogger('uvicorn.error')
    # logger = logging.getLogger(__name__)

    # find root and set share_dir
    logger.info(f"FELIX_IO running from: {Util.project_root()}")
    get_shared().share_dir = Util.project_root() / "share" / "felix-io"

    # Only here the settings of the environment are available and can be copied
    get_shared().simulate = get_settings().simulate

    if get_settings().cache > 0:
        get_shared().cache = Redis()  # Redis('/tmp/felix-io-redis.db')
        get_shared().cache_ttl = get_settings().cache
        logger.info(f"Using Redis cache for i2c calls on {get_shared().cache.db} with expiry of {get_shared().cache_ttl} seconds")

    get_shared().hostname = socket.gethostname()
    logger.info(f"Running on host: {get_shared().hostname}")

    config_dir = get_shared().share_dir / "config"
    logger.info(f"Found config in: {config_dir}")

    get_shared().driver = Driver(config_dir, get_shared().simulate)
    logger.info("Created driver for " + ("SIMULATED" if get_shared().simulate else "REAL") + " cards")

    get_shared().n_devices = get_shared().driver.number_of_devices()
    logger.info(f"Found {get_shared().n_devices} device(s)")

    for device_id in range(0, get_shared().n_devices):
        get_shared().driver.create_device(device_id)

    template_dir = get_shared().share_dir / "templates"
    get_shared().templates = Jinja2Templates(directory=template_dir)
    logger.info(f"Found templates in: {template_dir}")

    website_dir = get_shared().share_dir / "web"
    app.mount("/static", StaticFiles(directory=website_dir, html=True), name="static")
    logger.info(f"Found static website in: {website_dir}")

    if get_shared().n_devices == 0:
        logger.warning("\nCannot find ANY devices")
    else:
        # Simple test
        for device_id in range(0, get_shared().n_devices):
            logger.info(f"Device {device_id} card_type: {get_shared().driver.device(device_id).bitfield('CARD_TYPE').to_string()}")

    # Start clearing scheduler for latched registers
    scheduler = None
    if get_settings().clear:
        loop = asyncio.get_event_loop()
        scheduler = AsyncIOScheduler(event_loop=loop)
        for device_id in range(0, get_shared().n_devices):
            device = get_shared().driver.device(device_id)
            for bitfield_name in device.bitfield_names():
                clear_interval = device.bitfield(bitfield_name).clear_interval()
                if clear_interval > 0:
                    logger.info(f"{device_id}:{bitfield_name} cleared every {clear_interval} seconds")
                    scheduler.add_job(clear_latched_registers, "interval", [device_id, bitfield_name], seconds=clear_interval)
        scheduler.start()

    # Scan for aliases (i2c followed by regmap)
    for device_id in range(0, get_shared().n_devices):
        device = get_shared().driver.device(device_id)
        # Store alias for i2c
        for i2c_device_name in device.i2c().device_names():
            for i2c_register_name in device.i2c().register_names(i2c_device_name):
                alias = device.i2c().register(i2c_device_name, i2c_register_name).get('alias', None)
                if alias:
                    logger.debug(f"I2C Alias      '{alias}' found for {device_id}:{i2c_device_name}:{i2c_register_name}")
                    if device_id not in get_shared().alias:
                        get_shared().alias[device_id] = {}
                    if alias not in get_shared().alias[device_id]:
                        get_shared().alias[device_id][alias] = []
                    alias_list = get_shared().alias[device_id][alias]
                    alias_list.append(f'{i2c_device_name}:{i2c_register_name}')
        # Store aliases for regmap (normally defaults if i2c values do not exist)
        for bitfield_name in device.bitfield_names():
            alias = device.bitfield(bitfield_name).alias()
            if alias:
                logger.debug(f"Register Alias '{alias}' found for {device_id}:{bitfield_name}")
                if device_id not in get_shared().alias:
                    get_shared().alias[device_id] = {}
                if alias not in get_shared().alias[device_id]:
                    get_shared().alias[device_id][alias] = []
                alias_list = get_shared().alias[device_id][alias]
                alias_list.append(bitfield_name)

    # Print out which aliases were found
    logger.debug("ALIASES")
    for device_id in get_shared().alias:
        for alias in get_shared().alias[device_id]:
            alias_list = get_shared().alias[device_id][alias]
            logger.debug(f"Device {device_id} Alias '{alias}' -> {len(alias_list)}")
            for alias_index, alias_value in enumerate(alias_list):
                logger.debug(f"    {alias_index}: {alias_value}")
    logger.debug("")

    # Only aliases can be added to groups
    for device_id in range(0, get_shared().n_devices):
        device = get_shared().driver.device(device_id)

        # I2C
        i2c = device.i2c()
        for i2c_device_name in i2c.device_names():
            for i2c_register_name in i2c.register_names(i2c_device_name):
                register_values = i2c.register(i2c_device_name, i2c_register_name)
                groups = register_values.get('group', None)
                alias = register_values.get('alias', None)
                if groups and alias:
                    # print(card_type, i2c_device_name, i2c_register_name, register_values)
                    for group_name in groups if isinstance(groups, list) else (groups,):
                        if device_id not in get_shared().group:
                            get_shared().group[device_id] = {}
                        if group_name not in get_shared().group[device_id]:
                            get_shared().group[device_id][group_name] = set()
                        group = get_shared().group[device_id][group_name]
                        group.add(alias)

        for bitfield_name in device.bitfield_names():
            group_value = device.bitfield(bitfield_name).group()
            if group_value:
                for group_name in group_value if isinstance(group_value, list) else (group_value,):
                    if device_id not in get_shared().group:
                        get_shared().group[device_id] = {}
                    if group_name not in get_shared().group[device_id]:
                        get_shared().group[device_id][group_name] = set()
                    group = get_shared().group[device_id][group_name]
                    group.add(bitfield_name)

    # Print out which groups were found
    logger.debug("GROUPS")
    for device_id in get_shared().group:
        for group_name in get_shared().group[device_id]:
            group = get_shared().group[device_id][group_name]
            logger.debug(f"Device {device_id} Group '{group_name}' -> {len(group)}")
            for group_index, group_value in enumerate(group):
                logger.debug(f"    {group_index}: {group_value}")
    logger.debug("")

    app.include_router(api_router)
    app.include_router(support_router)
    app.include_router(list_router)
    app.include_router(js_router)
    app.include_router(data_router)

    yield

    # Stop
    logger.info("Shutdown")
    if scheduler:
        scheduler.shutdown()


app = FastAPI(
    dependencies=[Depends(get_settings), Depends(get_shared)],
    lifespan=lifespan,
    title=title,
    description=description,
    summary=summary,
    version=version,
)
