from typing import Annotated

# import markdown

from fastapi import APIRouter, Depends, Request
from fastapi.responses import HTMLResponse, RedirectResponse

from felix_io_api.api import Tags, get_shared  # , Shared
from .config import Settings, get_settings

#
# Support
#
support_router = APIRouter()


@support_router.get("/", response_class=HTMLResponse, tags=[Tags.support])
@support_router.get("/index.html", response_class=HTMLResponse, tags=[Tags.support])
async def home(shared: Annotated[dict, Depends(get_shared)],
               request: Request) -> str:
    return shared.templates.TemplateResponse(
        request=request, name="index.html", context={"simulate": shared.simulate, "hostname": shared.hostname}
    )

# Example, make sure you use a Template to refer to {{ data.text|safe }} to incklude the converted md
# @support_router.get("/x.html", response_class=HTMLResponse, tags=[Tags.support])
# async def home(shared: Annotated[dict, Depends(get_shared)],
#                request: Request) -> str:
#     template_dir = get_shared().share_dir / "templates"
#     with open(template_dir / "x.md", "r", encoding="utf-8") as input_file:
#         text = input_file.read()
#     html = markdown.markdown(text, extensions=['extra', 'toc'])

#     data = {
#         "title": "FELIX IO",
#         "text": html
#     }

#     return shared.templates.TemplateResponse("x.html", {"request": request, "data": data})


@support_router.get("/favicon.ico", tags=[Tags.support])
async def redirect():
    response = RedirectResponse(url='/static/atlas.ico')
    return response


@support_router.get("/settings", tags=[Tags.support])
async def settings(settings: Annotated[dict, Depends(get_settings)]
                   ) -> Settings:
    return settings


# @support_router.get("/shared", response_model_exclude=["driver"], tags=[Tags.support])
# async def settings(shared: Annotated[dict, Depends(get_shared)]
#                    ) -> Shared:
#     return shared


@support_router.get("/card_to_device_id/{card_id}", operation_id="card_to_device_id", tags=[Tags.support])
async def card_to_device_id(shared: Annotated[dict, Depends(get_shared)],
                            card_id: int
                            ) -> int:
    return shared.driver.card_to_device_id(card_id)


# for tests only
@support_router.put("/reset_statistics/device/{device_id}", operation_id="reset_statistics", tags=[Tags.support])
async def reset_statistics(shared: Annotated[dict, Depends(get_shared)],
                           device_id: int
                           ):
    device = shared.driver.device(device_id)
    device.card.reset_statistics(device_id)


# for tests only
@support_router.get("/number_of_opens/device/{device_id}", operation_id="number_of_opens", tags=[Tags.support])
async def number_of_opens(shared: Annotated[dict, Depends(get_shared)],
                          device_id: int
                          ) -> int:
    device = shared.driver.device(device_id)
    return device.number_of_opens()


# for tests only
@support_router.get("/number_of_locks/device/{device_id}", operation_id="number_of_locks", tags=[Tags.support])
async def number_of_locks(shared: Annotated[dict, Depends(get_shared)],
                          device_id: int
                          ) -> int:
    device = shared.driver.device(device_id)
    return device.number_of_locks()


# for tests only
@support_router.get("/number_of_bitfields/device/{device_id}", operation_id="number_of_bitfields", tags=[Tags.support])
async def number_of_bitfields(shared: Annotated[dict, Depends(get_shared)],
                              device_id: int
                              ) -> int:
    device = shared.driver.device(device_id)
    return len(device.bitfield_names())


# for tests only
@support_router.get("/number_of_bitfields/card/{card_id}", operation_id="number_of_bitfields_card", tags=[Tags.support])
async def number_of_bitfields_card(shared: Annotated[dict, Depends(get_shared)],
                                   card_id: int
                                   ) -> int:
    device_id = shared.driver.card_to_device_id(card_id)
    return await number_of_bitfields(shared, device_id)


# for tests only
@support_router.get("/number_of_i2cs/device/{device_id}", operation_id="number_of_i2cs", tags=[Tags.support])
async def number_of_i2cs(shared: Annotated[dict, Depends(get_shared)],
                         device_id: int
                         ) -> int:
    device = shared.driver.device(device_id)
    return len(device.i2c_names())


# for tests only
@support_router.get("/number_of_i2cs/card/{card_id}", operation_id="number_of_i2cs_card", tags=[Tags.support])
async def number_of_i2cs_card(shared: Annotated[dict, Depends(get_shared)],
                              card_id: int
                              ) -> int:
    device_id = shared.driver.card_to_device_id(card_id)
    return await number_of_i2cs(shared, device_id)


# for tests only
@support_router.get("/number_of_aliases/device/{device_id}", operation_id="number_of_aliases", tags=[Tags.support])
async def number_of_aliases(shared: Annotated[dict, Depends(get_shared)],
                            device_id: int
                            ) -> int:
    device = shared.driver.device(device_id)
    return len(device.alias_names())


# for tests only
@support_router.get("/number_of_aliases/card/{card_id}", operation_id="number_of_aliases_card", tags=[Tags.support])
async def number_of_aliases_card(shared: Annotated[dict, Depends(get_shared)],
                                 card_id: int
                                 ) -> int:
    device_id = shared.driver.card_to_device_id(card_id)
    return await number_of_aliases(shared, device_id)


# for tests only
@support_router.get("/number_of_groups/device/{device_id}", operation_id="number_of_groups", tags=[Tags.support])
async def number_of_groups(shared: Annotated[dict, Depends(get_shared)],
                           device_id: int
                           ) -> int:
    device = shared.driver.device(device_id)
    return len(device.group_names())


# for tests only
@support_router.get("/number_of_groups/card/{card_id}", operation_id="number_of_groups_card", tags=[Tags.support])
async def number_of_groups_card(shared: Annotated[dict, Depends(get_shared)],
                                card_id: int
                                ) -> int:
    device_id = shared.driver.card_to_device_id(card_id)
    return await number_of_groups(shared, device_id)
