from enum import Enum
from pathlib import Path
from typing import Annotated, NotRequired, Any
# Please use `typing_extensions.TypedDict` instead of `typing.TypedDict` on Python < 3.12.
from typing_extensions import TypedDict

from fastapi import APIRouter, Depends, HTTPException

from redislite import Redis


title = """FELIX-IO API"""
summary = f"""{title} allows you to interact with the felixcards in a host."""
version = "0.1"
description = f"""
{summary}

## API

Reads cards, devices, register bitfields and i2c devices.

## Support

Shows Devices and Bitfield values in tables.
"""


class Shared():
    hostname: str = None
    share_dir: Path = None
    simulate: bool = False
    cache: Redis = None
    cache_ttl: int = 0
    n_devices: int = 0
    # stored by device_id -> group -> set of aliases OR registernames (not i2c names)
    group: dict[int, dict[str, set[str]]] = {}
    templates: None
    # Driver = None, to decouple from  felix-io
    driver: None
    # stored by device_id -> alias -> set of bitfield_name OR i2c_device_name:i2c_register_name
    alias: dict[int, dict[str, set[str]]] = {}


shared = Shared()


def get_shared():
    return shared


class Tags(Enum):
    felix_io = "felix-io"
    support = "support"
    list = "list"  # noqa: A003
    javascript = "javascript"
    data = "data"


#
# Records for json
#
class BitFieldInfoRecord(TypedDict):
    desc: NotRequired[str]
    address: NotRequired[int]
    mask: NotRequired[int]
    bf_hi: NotRequired[int]
    bf_lo: NotRequired[int]
    fw_modes: NotRequired[str]
    type: NotRequired[str]  # noqa: A003
    endpoints: NotRequired[str]
    power: NotRequired[float]
    multiply: NotRequired[float]
    divide: NotRequired[float]
    offset: NotRequired[float]
    unit: NotRequired[str]
    clear: NotRequired[int]
    alias: NotRequired[str]
    # group: NotRequired[list[str] | str]


class BitFieldRecord(TypedDict):
    info: NotRequired[BitFieldInfoRecord]
    value: NotRequired[Any]
    raw_value: NotRequired[Any]
    unit: NotRequired[str]
    decoded_value: NotRequired[Any]
    ttl: NotRequired[int]


class DeviceRecord(TypedDict):
    name: str
    device_id: int
    card_id: int
    card_type: str


#
# API
#
api_router = APIRouter()


#
# Bitfield
#
@api_router.get("/api/device/{device_id}/bitfield/{name}", operation_id="read_bitfield", tags=[Tags.felix_io])
async def read_bitfield(shared: Annotated[dict, Depends(get_shared)],
                        device_id:  int,
                        name: str,
                        info: bool = False
                        ) -> dict[str, BitFieldRecord]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    json = device.read_bitfield(name, info)
    if not json:
        raise HTTPException(status_code=404, detail=f"Bitfield {name} not found for device {device_id}")

    return json


@api_router.get("/api/card/{card_id}/bitfield/{name}", operation_id="read_card_bitfield", tags=[Tags.felix_io])
async def read_card_bitfield(shared: Annotated[dict, Depends(get_shared)],
                             card_id: int,
                             name: str,
                             info: bool = False
                             ) -> dict[str, BitFieldRecord]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await read_bitfield(shared, device_id, name, info)


@api_router.get("/api/device/{device_id}/bitfield", operation_id="list_bitfields", tags=[Tags.felix_io])
async def list_bitfields(shared: Annotated[dict, Depends(get_shared)],
                         device_id:  int
                         ) -> list[str]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    return device.bitfield_names()


@api_router.get("/api/card/{card_id}/bitfield", operation_id="list_card_bitfields", tags=[Tags.felix_io])
async def list_card_bitfields(shared: Annotated[dict, Depends(get_shared)],
                              card_id:  int
                              ) -> list[str]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await list_bitfields(shared, device_id)


@api_router.put("/api/device/{device_id}/bitfield/{name}", operation_id="write_bitfield", tags=[Tags.felix_io])
async def write_bitfield(shared: Annotated[dict, Depends(get_shared)],
                         device_id: int,
                         name: str,
                         value: int,
                         info: bool = False
                         ) -> dict[str, BitFieldRecord]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    json = device.write_bitfield(name, value, info)
    if not json:
        raise HTTPException(status_code=404, detail=f"Writeable Bitfield {name} not found for device {device_id}")

    return json


@api_router.put("/api/card/{card_id}/bitfield/{name}", operation_id="write_card_bitfield", tags=[Tags.felix_io])
async def write_card_bitfield(shared: Annotated[dict, Depends(get_shared)],
                              card_id: int,
                              name: str,
                              value: int,
                              info: bool = False
                              ) -> dict[str, BitFieldRecord]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await write_bitfield(shared, device_id, name, value, info)


#
# I2C
#
@api_router.get("/api/device/{device_id}/i2c/{name}", operation_id="read_i2c", tags=[Tags.felix_io])
async def read_i2c(shared: Annotated[dict, Depends(get_shared)],
                   device_id: int,
                   name: str,
                   info: bool = False
                   ) -> dict[str, BitFieldRecord]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    json = device.read_i2c(name, info)
    if not json:
        raise HTTPException(status_code=404, detail=f"I2C {name} not found for device {device_id}")

    return json


@api_router.get("/api/card/{card_id}/i2c/{name}", operation_id="read_card_i2c", tags=[Tags.felix_io])
async def read_card_i2c(shared: Annotated[dict, Depends(get_shared)],
                        card_id: int,
                        name: str,
                        info: bool = False
                        ) -> dict[str, BitFieldRecord]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await read_i2c(shared, device_id, name, info)


@api_router.get("/api/device/{device_id}/i2c", operation_id="list_i2c", tags=[Tags.felix_io])
async def list_i2c(shared: Annotated[dict, Depends(get_shared)],
                   device_id:  int
                   ) -> list[str]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    return device.i2c_names()


@api_router.get("/api/card/{card_id}/i2c", operation_id="list_card_i2c", tags=[Tags.felix_io])
async def list_card_i2c(shared: Annotated[dict, Depends(get_shared)],
                              card_id:  int
                              ) -> list[str]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await list_i2c(shared, device_id)


#
# Alias
#
@api_router.get("/api/device/{device_id}/alias/{name}", operation_id="read_alias", tags=[Tags.felix_io])
async def read_alias(shared: Annotated[dict, Depends(get_shared)],
                     device_id: int,
                     name: str,
                     info: bool = False
                     ) -> dict[str, BitFieldRecord]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    json = device.read_alias(name, info)
    if not json:
        raise HTTPException(status_code=404, detail=f"Alias {name} not found for device {device_id}")

    return json


@api_router.get("/api/card/{card_id}/alias/{name}", operation_id="read_card_alias", tags=[Tags.felix_io])
async def read_card_alias(shared: Annotated[dict, Depends(get_shared)],
                          card_id: int,
                          name: str,
                          info: bool = False
                          ) -> dict[str, BitFieldRecord]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await read_alias(shared, device_id, name, info)


@api_router.get("/api/device/{device_id}/alias", operation_id="list_aliases", tags=[Tags.felix_io])
async def list_aliases(shared: Annotated[dict, Depends(get_shared)],
                   device_id:  int
                   ) -> list[str]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    return device.alias_names()


@api_router.get("/api/card/{card_id}/alias", operation_id="list_card_aliases", tags=[Tags.felix_io])
async def list_card_aliases(shared: Annotated[dict, Depends(get_shared)],
                              card_id:  int
                              ) -> list[str]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await list_aliases(shared, device_id)


#
# Group
#
@api_router.get("/api/device/{device_id}/group/{name}", operation_id="read_group", tags=[Tags.felix_io])
async def read_group(shared: Annotated[dict, Depends(get_shared)],
                     device_id:  int,
                     name: str,
                     info: bool = False
                     ) -> dict[str, BitFieldRecord]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    json = device.read_group(name, info)
    if not json:
        raise HTTPException(status_code=404, detail=f"Group {name} not found for device {device_id}")
    return json


@api_router.get("/api/card/{card_id}/group/{name}", operation_id="read_card_group", tags=[Tags.felix_io])
async def read_card_group(shared: Annotated[dict, Depends(get_shared)],
                          card_id:  int,
                          name: str,
                          info: bool = False
                          ) -> dict[str, BitFieldRecord]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await read_group(shared, device_id, name, info)


@api_router.get("/api/device/{device_id}/group", operation_id="list_groups", tags=[Tags.felix_io])
async def list_groups(shared: Annotated[dict, Depends(get_shared)],
                      device_id:  int
                      ) -> list[str]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    return device.group_names()


@api_router.get("/api/card/{card_id}/group", operation_id="list_card_groups", tags=[Tags.felix_io])
async def list_card_groups(shared: Annotated[dict, Depends(get_shared)],
                           card_id:  int
                           ) -> list[str]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await list_groups(shared, device_id)


#
# API
#
@api_router.get("/api", operation_id="list_devices", tags=[Tags.felix_io])
async def list_devices(shared: Annotated[dict, Depends(get_shared)]) -> list[DeviceRecord]:
    return shared.driver.device_list()


@api_router.get("/api/device/{device_id}/{name}", operation_id="read", tags=[Tags.felix_io])
async def read(shared: Annotated[dict, Depends(get_shared)],
               device_id:  int,
               name: str,
               info: bool = False
               ) -> dict[str, BitFieldRecord]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    json = device.read(name, info)
    if not json:
        raise HTTPException(status_code=404, detail=f"{name} not found for device {device_id}")
    return json


@api_router.get("/api/card/{card_id}/{name}", operation_id="read_card", tags=[Tags.felix_io])
async def read_card(shared: Annotated[dict, Depends(get_shared)],
                    card_id:  int,
                    name: str,
                    info: bool = False
                    ) -> dict[str, BitFieldRecord]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await read(shared, device_id, name, info)


#
# POST
#
@api_router.post("/api/device/{device_id}", operation_id="read_list", tags=[Tags.felix_io])
async def read_list(shared: Annotated[dict, Depends(get_shared)],
                    device_id:  int,
                    names: list[str],
                    info: bool = False
                    ) -> dict[str, BitFieldRecord]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    json = device.read_list(names, info)
    if not json:
        raise HTTPException(status_code=404, detail=f"Cannot read list of bitfields for device {device_id}")
    return json


@api_router.post("/api/card/{card_id}", operation_id="read_list_card", tags=[Tags.felix_io])
async def read_list_card(shared: Annotated[dict, Depends(get_shared)],
                         card_id: int,
                         names: list[str],
                         info: bool = False
                         ) -> dict[str, BitFieldRecord]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await read_list(shared, device_id, names, info)


#
# PUT
#
@api_router.put("/api/device/{device_id}/{name}", operation_id="write", tags=[Tags.felix_io])
async def write(shared: Annotated[dict, Depends(get_shared)],
                device_id: int,
                name: str,
                value: int,
                info: bool = False
                ) -> dict[str, BitFieldRecord]:
    device = shared.driver.device(device_id)
    if not device:
        raise HTTPException(status_code=404, detail=f"Device {device_id} not found")

    json = device.write(name, value, info)
    if not json:
        raise HTTPException(status_code=404, detail=f"Writeable {name} not found for device {device_id}")
    return json


@api_router.put("/api/card/{card_id}/{name}", operation_id="write_card", tags=[Tags.felix_io])
async def write_card(shared: Annotated[dict, Depends(get_shared)],
                     card_id: int,
                     name: str,
                     value: int,
                     info: bool = False
                     ) -> dict[str, BitFieldRecord]:
    device_id = shared.driver.card_to_device_id(card_id)
    return await write(shared, device_id, name, value, info)
