cmake_minimum_required(VERSION 3.21)

macro(felix_setup)
  message(STATUS "Project ${PROJECT_NAME} is top level: ${PROJECT_IS_TOP_LEVEL}")

  include(ProjectOptions)
  felix_setup_options()

  if(PROJECT_IS_TOP_LEVEL)
    include(Policies)
    felix_policies()
    include(FELIX-versions)

    include(CTest)

    include(TDAQ)
    if(NOLCG)
      tdaq_project(${PACKAGE} ${PACKAGE_VERSION} )
    else()
      if(DEFINED ENV{LCG_VERSION_POSTFIX})
        set(LCG_VERSION_POSTFIX $ENV{LCG_VERSION_POSTFIX})
      else()
        set(LCG_VERSION_POSTFIX "a")
      endif()
      if(DEFINED ENV{LCG_RELEASE_BASE})
        set(LCG_RELEASE_BASE $ENV{LCG_RELEASE_BASE})
      endif()
      tdaq_project(${PACKAGE} ${PACKAGE_VERSION} USES LCG $ENV{LCG_VERSION})
      unset(ROOT_ROOT CACHE)  # Do not allow anyone to find this evil piece of software
    endif()

    felix_global_options()
    include(Dependencies)
  endif()
  felix_local_options()

  #
  # sub project specific
  #
  if(NOT ${PACKAGE} EQUAL "felix")
    tdaq_package(NO_HEADERS)
    project(${PACKAGE})
    remove_definitions_with_prefix("TDAQ_PACKAGE_NAME")
  endif()
endmacro()

function(create_install_script)
  if(NOT PROJECT_IS_TOP_LEVEL AND NOT FELIX_INSTALL_ALL)
    return()
  endif()
  find_file(setup_in NAMES setup_felix.sh.in HINTS ${CMAKE_MODULE_PATH} PATH_SUFFIXES templates NO_DEFAULT_PATH)
  if(setup_in)
    message(STATUS "Found ${setup_in} template, generating")

    configure_file(${setup_in} setup_felix.sh @ONLY)
    install(FILES ${CMAKE_CURRENT_BINARY_DIR}/setup_felix.sh OPTIONAL COMPONENT ${BINARY_TAG} DESTINATION .)
  else()
    message(STATUS "Unable to find ${setup_in} template in ${CMAKE_MODULE_PATH}")
  endif()
endfunction(create_install_script)

function(felix_add_executable target)
  set(options ALWAYS_INSTALL TEST)
  cmake_parse_arguments(ARG "${options}" "" "" ${ARGN})
  set(destination "bin")
  if(ARG_TEST)
    set(destination "test/${PACKAGE}")
  endif()

  if(PROJECT_IS_TOP_LEVEL OR ARG_ALWAYS_INSTALL OR FELIX_INSTALL_ALL)
    tdaq_add_executable(${target} DESTINATION ${destination} ${ARG_UNPARSED_ARGUMENTS})
  else()
    tdaq_add_executable(${target} ${ARG_UNPARSED_ARGUMENTS} NOINSTALL)
  endif()

  target_link_libraries(${target} PRIVATE ${PROJECT_NAME}_warnings ${PROJECT_NAME}_options)
  # _REENTRANT changes binary layout and is added by TDAQ. It therefore has to be added to our libraries and all targets depending on them
  target_compile_definitions(${target} PRIVATE -D_GNU_SOURCE -D_REENTRANT -D__USE_XOPEN2K8)
endfunction()

function(felix_add_library target)
  set(options)
  set(oneValueArgs EXPORT)
  set(multiValueArgs)
  cmake_parse_arguments(ARG "${options}" "${oneValueArgs}" "${multiValueArgs}" ${ARGN})

  if(ARG_EXPORT)
    tdaq_add_library(${target} DESTINATION lib ${ARG_UNPARSED_ARGUMENTS} NOINSTALL)
    install(TARGETS ${target} COMPONENT ${TDAQ_COMPONENT_BINARY} EXPORT ${ARG_EXPORT} DESTINATION lib)
    install(FILES ${CMAKE_CURRENT_BINARY_DIR}/lib${target}.so.debug OPTIONAL COMPONENT ${TDAQ_COMPONENT_BINARY} DESTINATION lib/.debug)
  else()
    tdaq_add_library(${target} DESTINATION lib ${ARG_UNPARSED_ARGUMENTS})
  endif()

  target_link_libraries(${target} PRIVATE ${PROJECT_NAME}_warnings ${PROJECT_NAME}_options)
  # _REENTRANT changes binary layout and is added by TDAQ. It therefore has to be added to our libraries and all targets depending on them
  target_compile_definitions(${target} PUBLIC -D_GNU_SOURCE -D_REENTRANT -D__USE_XOPEN2K8)
endfunction()

function(felix_add_test target file)
  if(NOT file)
    message(FATAL_ERROR "No test file specified for ${target}")
  endif()

  set(python_path $ENV{PYTHONPATH})

  # Check if python_env directory exists
  if(EXISTS "${CMAKE_SOURCE_DIR}/python_env")
    file(GLOB site_packages_dir "${CMAKE_SOURCE_DIR}/python_env/lib/*/site-packages")
    if(site_packages_dir)
      list(GET site_packages_dir 0 site_packages_dir)  # Get the first match
      set(python_path "${site_packages_dir}:${python_path}")
    endif()
  endif()

  foreach(package ${felix_internal_packages})
    if(DEFINED ${package}_BINARY_DIR)
      set(python_path ${${package}_BINARY_DIR}:${python_path})
    endif()
  endforeach()

  add_test(NAME ${target} COMMAND ${file} ${ARGN})
  set_tests_properties(${target} PROPERTIES ENVIRONMENT "PYTHONPATH=${python_path}")
endfunction()

function(felix_install_headers)
  cmake_parse_arguments(ARG "" "DESTINATION" "" ${ARGN})
  felix_install_directory(
    ${ARGN}
    ALWAYS_INSTALL  # Always install headers
    DESTINATION "include/${ARG_DESTINATION}"
    DEFAULT_PATTERN "*.h;*.hh;*.hpp"
  )
endfunction()

function(felix_install_python)
  cmake_parse_arguments(ARG "" "DESTINATION" "" ${ARGN})
  felix_install_files(
    ${ARGN}
    DESTINATION "python/${ARG_DESTINATION}"
    DEFAULT_PATTERN "*.py"
  )
endfunction()

function(felix_install_shared)
  cmake_parse_arguments(ARG "" "DESTINATION" "" ${ARGN})
  felix_install_files(
    ${ARGN}
    DESTINATION "share/${ARG_DESTINATION}"
    DEFAULT_PATTERN "*"
  )
endfunction()

function(felix_install_directory)
  cmake_parse_arguments(ARG "ALWAYS_INSTALL" "DESTINATION" "FILES;DEFAULT_PATTERN" ${ARGN})
  if(NOT PROJECT_IS_TOP_LEVEL AND NOT ARG_ALWAYS_INSTALL AND NOT FELIX_INSTALL_ALL)
    return()
  endif()

  foreach(f ${ARG_FILES})
    set(pattern ${pattern} PATTERN ${f})
  endforeach()

  if(NOT ARG_FILES)
    foreach(p ${ARG_DEFAULT_PATTERN})
      set(pattern ${pattern} PATTERN ${p})
    endforeach()
  endif()

  foreach(d ${ARG_UNPARSED_ARGUMENTS})
    install(DIRECTORY ${d} OPTIONAL COMPONENT ${TDAQ_COMPONENT_NOARCH} DESTINATION ${ARG_DESTINATION} FILES_MATCHING ${pattern} PATTERN .git EXCLUDE)
  endforeach()
endfunction()

function(felix_install_files)
  cmake_parse_arguments(ARG "ALWAYS_INSTALL;MAKE_EXECUTABLE" "DESTINATION;DEFAULT_PATTERN" "FILES" ${ARGN})
  if(NOT PROJECT_IS_TOP_LEVEL AND NOT ARG_ALWAYS_INSTALL AND NOT FELIX_INSTALL_ALL)
    return()
  endif()

  if(NOT ARG_FILES)
    set(ARG_FILES ${ARG_DEFAULT_PATTERN})
  endif()

  foreach(d ${ARG_UNPARSED_ARGUMENTS})
    foreach(f ${ARG_FILES})
      file(GLOB_RECURSE matched_files "${d}/${f}")

      foreach(file ${matched_files})
        if (ARG_MAKE_EXECUTABLE)
          install(
            FILES ${file}
            OPTIONAL
            COMPONENT ${TDAQ_COMPONENT_NOARCH}
            DESTINATION ${ARG_DESTINATION}
            PERMISSIONS
                OWNER_READ OWNER_WRITE OWNER_EXECUTE
                GROUP_READ GROUP_EXECUTE
                WORLD_READ WORLD_EXECUTE
          )
        else()
          install(FILES ${file} OPTIONAL COMPONENT ${TDAQ_COMPONENT_NOARCH} DESTINATION ${ARG_DESTINATION})
        endif()
      endforeach()
    endforeach()
  endforeach()
endfunction()
