import argparse
import difflib
import logging
import os
import urllib.request


def main():
    # Parse command line arguments
    parser = argparse.ArgumentParser(description='Download and compare CMake versions file.')
    parser.add_argument('--url', type=str, required=True, help='Base url of the repository')
    parser.add_argument('--branch', type=str, required=True, help='Branch of the repository')
    parser.add_argument('--existing-file-path', type=str, required=True, help='Path to the existing versions.cmake file')
    parser.add_argument('--force', action='store_true', help='Force overwrite the existing file even if there are changes')
    args = parser.parse_args()

    # URL of the file to download
    url = f"{args.url}/-/raw/{args.branch}/cmake/versions.cmake"

    # Download the file
    try:
        response = urllib.request.urlopen(url)
        downloaded_file_content = response.read().decode('utf-8')
    except Exception as e:
        logging.fatal(f"Failed to download versions file: {e}")
        return

    # If the existing file exists, compare it with the downloaded file
    if os.path.exists(args.existing_file_path):
        with open(args.existing_file_path, 'r') as file:
            existing_file_content = file.read()

        diff = difflib.unified_diff(existing_file_content.split('\n'), downloaded_file_content.split('\n'))
        changes = list(diff)

        if changes and not args.force:
            logging.warning("The existing versions.cmake file is outdated. Set OVERWRITE_VERSIONS to ON to overwrite.")
            print('\n'.join(changes))
            return
        if changes:
            logging.info("Overwriting the existing file due to changes.")
            print('\n'.join(changes))

    # Overwrite the existing file with the downloaded file
    with open(args.existing_file_path, "w") as file:
        file.write(downloaded_file_content)

if __name__ == "__main__":
    main()
