"""
Asciidoc Generator.

Usage:
  gen.py [options]

Options:
  --no-pdf                    Do not write PDF
  --no-toc -n                 Don't write TOC
  --config <name>, -c <name>  Config file name [default: config.json]
  --io <encoding>             IO Encoding [default: utf-8]
  --version                   Show version of the tool
"""
import fileinput
import json
import os
import re
import shutil
import sys

from docopt import docopt
from subprocess import call

VERSION = "Asciidoc Generator 2.3"

io_encoding = ""
no_pdf = False

config = {}
in_dir = ""
out_dir = ""

bibliography = {}

titles = []
toc_level = 0

search_data = []
search_id = -1
recent_item = None


def add_search_index(title, filename):
    """TBD."""
    global search_data
    global search_id
    global recent_item
    if recent_item:
        search_data.append(recent_item)
        recent_item = None
    if title:
        search_id += 1
        recent_item = {"id": search_id, "title": title, "body": "", "filename": filename}


def read_config(config_name):
    """TBD."""
    global config
    global in_dir
    global out_dir
    global toc_level
    # load the config file
    with open(config_name, 'r', encoding=io_encoding) as configFile:
        config = json.loads(configFile.read())
    toc_level = config["toc_max_level"]
    out_dir = config["output_dir"]
    in_dir = config["input_dir"]


def clean_output():
    """TBD."""
    # initialize the output directory
    if os.path.exists(out_dir):
        shutil.rmtree(out_dir)
    os.makedirs(out_dir)


def id_name_register(matchobj):
    """# register [[[id]]] or [[[id,name]]] and replace by [[[id,id]]id] or [[[id,name]]name]."""
    global bibliography

    id = matchobj.group("id")
    name = matchobj.group("name")
    name = id if name is None else name
    bibliography[id] = name

    return "[[[" + id + "," + name + "]]" + name + "]"


def ref_name_repl(matchobj):
    """Replace <<bibliography#ref>> into <<bibliography#ref, name>>."""
    global bibliography

    ref = matchobj.group('ref')
    if ref in bibliography:
        name = bibliography[ref]
    else:
        name = ref
        sys.stderr.write("ERROR: " + ref + " not found in bibliography\n")
    return "<<bibliography#" + ref + "," + name + ">>"


def main():
    """TBD."""
    global no_pdf
    global bibliography

    print('Handling bibliography')
    bibliography = {}
    try:
        for line in fileinput.input(in_dir + 'bibliography.adoc', inplace=True):
            # register [[[id]]] or [[[id,name]]] and replace by [[[id]]id] or [[[id,name]]name]
            line = re.sub(r'\[\[\[(?P<id>[^,]*)(,\s*(?P<name>.+))?\]\]\]', id_name_register, line)
            print(line, end="")
    except UnicodeDecodeError:
        sys.stderr.write("ERROR: " + 'bibliography.adoc' + ":" + str(fileinput.lineno()) + " contains non-ascii code")
    fileinput.close()

    for entry in config["files"]:
        print("Handling " + entry["filename"])

        # handle --no-pdf
        if "type" in entry and entry["type"] == "pdf" and no_pdf:
            continue

        # Fixup bibliography refs
        try:
            for line in fileinput.input(in_dir + entry["filename"], inplace=True):
                # look for hardcoded references
                for ref_match in re.finditer(r'<<bibliography#(?P<ref>[^,]*?),\s*.*?>>', line):
                    sys.stderr.write("WARNING: in '" + entry["filename"] + ":" + str(fileinput.lineno()) + "' reference '" + ref_match.group("ref") + "' is hardcoded\n")

                # replace <<bibliography#ref>> into <<bibliography#ref, name>>
                line = re.sub(r'<<bibliography#(?P<ref>[^,]*?)>>', ref_name_repl, line)
                print(line, end="")
        except UnicodeDecodeError:
            sys.stderr.write("ERROR: " + entry["filename"] + ":" + str(fileinput.lineno()) + " contains non-ascii code")
        fileinput.close()

        # run asciidocter
        params = ["asciidoctor"]
        params.append("--destination-dir=" + out_dir)
        params.append("--doctype=book")
        params.append("--require=asciidoctor-pdf")
        params.append("--attribute=last-update-label!")
        params.append("--attribute=icons=font")
        params.append("--attribute=xrefstyle=short")
        if "format" in entry:
            params.append("--backend=" + entry["format"])

        params.append(in_dir + entry["filename"])
        print(' '.join(params))
        call(params, shell=(os.name == "nt"))

        if "alternative_pdf_pp" in entry and entry["alternative_pdf_pp"] is True:
            params = ["fopub"]
            params.append(out_dir + entry["filename"][:entry["filename"].rfind(".")] + ".xml")
            print(' '.join(params))
            call(params, shell=(os.name == "nt"))

        if "exclude_toc" in entry and entry["exclude_toc"] is True:
            continue

        with open(in_dir + entry["filename"], "r", encoding=io_encoding) as file:
            lines = file.readlines()
        last_id = ("", -2)
        index = 0
        for line in lines:
            # match: '[#sec:introduction]' --> id: 'sec:introduction'
            id_match = re.match(r"\[#(?P<id>.+)\]", line)
            if id_match:
                last_id = (id_match.group("id"), index)

            # match: '= FELIX Variants and Functionality' --> eq: '=', title: 'FELIX Variants and Functionality'
            # match: '== Gigabit Transceiver (GBT) and the Versatile Link' --> eq: '==', title: 'Gigabit Transceiver (GBT) and the Versatile Link'
            title_match = re.match(r"(?<!\s)(?P<eq>={1,6}) (?P<title>.+)", line)
            if title_match:
                lvl = len(title_match.group("eq"))
                if (lvl > toc_level):
                    continue
                title = title_match.group("title")

                id = ""
                if last_id[1] == index - 1:
                    id = last_id[0]
                else:
                    id = "_" + re.sub(r"\W", "_", title.lower())
                filename = "/" + entry["filename"]
                titles.append((lvl, filename + "#" + id, title))

                add_search_index(title, filename.replace(".adoc", ".html") + "#" + id)
            elif recent_item:
                recent_item["body"] += line

            index += 1
        add_search_index(None, None)

    with open(out_dir + config["search_index_file"], "w", encoding=io_encoding) as index_file:
        index_file.write(json.dumps(search_data))


def write_toc():
    """TBD."""
    with open(in_dir + config["toc_file"], "w", encoding=io_encoding) as toc_file:
        toc_file.write("= Index\n\n")
        for heading in titles:
            toc_file.write(("." * heading[0]) + " <<" + heading[1] + "," + heading[2] + ">>\n")

    params = ["asciidoctor"]
    params.append("-D" + out_dir)
    params.append(in_dir + config["toc_file"])
    call(params, shell=(os.name == "nt"))


if __name__ == "__main__":
    """TBD."""
    args = docopt(__doc__, version=VERSION)
    io_encoding = args["--io"]
    no_pdf = args["--no-pdf"]

    read_config(args["--config"])
    clean_output()

    if "pre_cmd" in config and os.name in config["pre_cmd"]:
        for cmd in config["pre_cmd"][os.name]:
            call(cmd, shell=True)

    main()

    if not args["--no-toc"]:
        write_toc()

    if "post_cmd" in config and os.name in config["post_cmd"]:
        for cmd in config["post_cmd"][os.name]:
            call(cmd, shell=True)
