#!/usr/bin/env python3
"""
FELIX Doctool - Asciidoc to html and pdf converter.

Usage:
  felix-doctool.py [options]

Options:
  --no-pdf                    Do not write PDF
  --config <name>, -c <name>  Config file name [default: config.json]
  --io <encoding>             IO Encoding [default: utf-8]
  --version                   Show version of the tool
"""

__author__ = "Mark Donszelmann <Mark.Donszelmann@cern.ch>"
__version__ = '3.0'

import fileinput
import json
import os
import re
import shutil
import sys

from collections import namedtuple
from docopt import docopt
from subprocess import call

io_encoding = ""
no_pdf = False

config = {}
in_dir = ""
out_dir = ""

bibliography = {}

Title = namedtuple('Title', 'level section ref title')
titles = []
toc_level = 0

current_level = []
last_id = ()
index = 0
entry = {}

search_data = []
search_id = -1
recent_item = None

__VERSION__ = os.path.splitext(os.path.basename(__file__))[0] + ' ' + __version__


def add_search_index(title, filename):
    """TBD."""
    global search_data
    global search_id
    global recent_item
    if recent_item:
        search_data.append(recent_item)
        recent_item = None
    if title:
        search_id += 1
        recent_item = {"id": search_id, "title": title, "body": "", "filename": filename}


def read_config(config_name):
    """TBD."""
    global config
    global in_dir
    global out_dir
    global toc_level
    # load the config file
    with open(config_name, 'r', encoding=io_encoding) as configFile:
        config = json.loads(configFile.read())
    toc_level = config["toc_max_level"]
    out_dir = config["output_dir"]
    in_dir = config["input_dir"]


def clean_output():
    """TBD."""
    # initialize the output directory
    if os.path.exists(out_dir):
        shutil.rmtree(out_dir)
    os.makedirs(out_dir)


def id_name_register(matchobj):
    """# register [[[id]]] or [[[id,name]]] and replace by [[[id,id]]id] or [[[id,name]]name]."""
    global bibliography

    id = matchobj.group("id")
    name = matchobj.group("name")
    name = id if name is None else name
    bibliography[id] = name

    return "[[[" + id + "," + name + "]]" + name + "]"


def ref_name_repl(matchobj):
    """Replace <<bibliography#ref>> into <<bibliography#ref, name>>."""
    global bibliography

    ref = matchobj.group('ref')
    if ref in bibliography:
        name = bibliography[ref]
    else:
        name = ref
        sys.stderr.write("ERROR: " + ref + " not found in bibliography\n")
    return "[<<bibliography#" + ref + "," + name + ">>]"


def heading_replace(matchobj):
    """Replace '= FELIX Variants and Functionality' with '= 1. FELIX Variants and Functionality'."""
    global toc_level
    global current_level
    global last_id
    global titles
    global index
    global entry

    level = len(matchobj.group("eq"))
    section = None
    if (level <= toc_level):

        # adjust current level according to level
        current_level[level - 1] += 1
        for i in range(level, toc_level):
            current_level[i] = 0

        # create section label
        section = ''
        for i in range(level):
            section += str(current_level[i]) + ('.' if i == 0 or i < level - 1 else '')

        title = matchobj.group("title")

        id = ""
        if last_id[1] == index - 1:
            id = last_id[0]
        else:
            id = title
            id = re.sub(r"/", "", id)
            id = re.sub(r"\(C\)", "", id)
            id = re.sub(r"\(R\)", "", id)
            id = re.sub(r"\(TM\)", "", id)
            id = id.lower()
            id = re.sub(r"\W+", "_", id)
            id = "_" + section.replace('.', '_') + "_" + id
            id = id.rstrip('_')
            # NOTE: remove trailing underscore
        filename = entry["filename"]
        titles.append(Title(level, section, filename + "#" + id, title))

        add_search_index(title, filename.replace(".adoc", ".html") + "#" + id)

    return matchobj.group("eq") + ' ' + ('' if section is None else section + ' ') + matchobj.group("title")


def fixup_adoc():
    """Look up search items in adoc and fix headers."""
    global current_level
    global in_dir
    global toc_level
    global last_id
    global index
    global entry

    # with open(in_dir + entry["filename"], "r", encoding=io_encoding) as file:
    #     lines = file.readlines()
    last_id = ("", -2)
    index = 0
    for line in fileinput.input(in_dir + entry["filename"], inplace=True):
        # match: '[#sec:introduction]' --> id: 'sec:introduction'
        id_match = re.match(r"\[#(?P<id>.+)\]", line)
        if id_match:
            last_id = (id_match.group("id"), index)

        # match: '= FELIX Variants and Functionality' --> eq: '=', title: 'FELIX Variants and Functionality'
        # match: '== Gigabit Transceiver (GBT) and the Versatile Link' --> eq: '==', title: 'Gigabit Transceiver (GBT) and the Versatile Link'
        line = re.sub(r"^(?P<eq>={1,6}) (?P<title>.+)", heading_replace, line)
        index += 1
        print(line, end="")

        # elif recent_item:
        #    recent_item["body"] += line
    add_search_index(None, None)


def main():
    """TBD."""
    global no_pdf
    global bibliography
    global current_level
    global entry

    print('Handling bibliography')
    bibliography = {}
    try:
        for line in fileinput.input(in_dir + 'bibliography.adoc', inplace=True):
            # register [[[id]]] or [[[id,name]]] and replace by [[[id]]id] or [[[id,name]]name]
            line = re.sub(r'\[\[\[(?P<id>[^,]*)(,\s*(?P<name>.+))?\]\]\]', id_name_register, line)
            print(line, end="")
    except UnicodeDecodeError:
        sys.stderr.write("ERROR: " + 'bibliography.adoc' + ":" + str(fileinput.lineno()) + " contains non-ascii code")
    fileinput.close()

    current_level = []
    for i in range(toc_level):
        current_level.append(0)

    for entry in config["files"]:
        print("Handling " + entry["filename"])

        # handle --no-pdf
        if "type" in entry and entry["type"] == "pdf" and no_pdf:
            continue

        # Fixup bibliography refs
        try:
            for line in fileinput.input(in_dir + entry["filename"], inplace=True):
                # look for hardcoded references
                for ref_match in re.finditer(r'<<bibliography#(?P<ref>[^,]*?),\s*.*?>>', line):
                    sys.stderr.write("WARNING: in '" + entry["filename"] + ":" + str(fileinput.lineno()) + "' reference '" + ref_match.group("ref") + "' is hardcoded\n")

                # replace <<bibliography#ref>> into <<bibliography#ref, name>>
                line = re.sub(r'<<bibliography#(?P<ref>[^,]*?)>>', ref_name_repl, line)
                print(line, end="")
        except UnicodeDecodeError:
            sys.stderr.write("ERROR: " + entry["filename"] + ":" + str(fileinput.lineno()) + " contains non-ascii code")
        fileinput.close()

        if "exclude_toc" not in entry or entry["exclude_toc"] is False:
            fixup_adoc()

        pdf = 'format' in entry and entry['format'] == 'pdf'

        # run asciidocter
        params = ["asciidoctor"]
        params.append("--destination-dir=" + out_dir)
        params.append("--doctype=book")
        params.append("--require=asciidoctor-pdf")
        params.append("--attribute=last-update-label!")
        params.append("--attribute=icons=font")
        params.append("--attribute=xrefstyle=short")
        params.append("--attribute=linkcss")
        params.append("--attribute=sectlinks")
        params.append("--attribute=sectanchors")
        if not pdf:
            params.append("--no-header-footer")
        if "format" in entry:
            params.append("--backend=" + entry["format"])
        params.append(in_dir + entry["filename"])
        print(' '.join(params))
        call(params, shell=(os.name == "nt"))

        if "type" not in entry or entry["type"] != "pdf":
            # Add the frontmatter to HTML
            for line in fileinput.input(out_dir + entry["filename"][:entry["filename"].rfind(".")] + ".html", inplace=True):
                if fileinput.lineno() == 1:
                    print("---")
                    print("layout: manual")
                    print("generator: " + __VERSION__)
                    # print("title: " + entry["filename"])
                    print("---")
                    front_matter = True
                print(line, end="")
            fileinput.close()

    with open(out_dir + config["search_index_file"], "w", encoding=io_encoding) as index_file:
        index_file.write(json.dumps(search_data))


def write_toc():
    """TBD."""
    with open(in_dir + config["toc_file"], "w", encoding=io_encoding) as toc_file:
        toc_file.write("= Index\n\n")
        toc_file.write("[#toc.toc2]\n")
        # toc_file.write("Table of Contents\n")
        current_level = 0
        for heading in titles:
            if heading[0] > current_level:
                toc_file.write("[sectlevel" + str(heading[0]) + "]\n")
            current_level = heading[0]
            toc_file.write(("*" * heading.level) + " <<" + heading.ref + "," + heading.section + " " + heading.title + ">>\n")

    params = ["asciidoctor"]
    params.append("-D" + out_dir)
    params.append(in_dir + config["toc_file"])
    params.append("--attribute=linkcss")
    params.append("--no-header-footer")
    call(params, shell=(os.name == "nt"))


if __name__ == "__main__":
    """TBD."""
    if sys.version_info < (3, 0, 0):
        sys.stderr.write("You need python 3.0 or later to run this script\n")
        exit(1)

    args = docopt(__doc__, version=__VERSION__)
    io_encoding = args["--io"]
    no_pdf = args["--no-pdf"]

    read_config(args["--config"])
    clean_output()

    if "pre_cmd" in config and os.name in config["pre_cmd"]:
        print("Running pre-cmd")
        for cmd in config["pre_cmd"][os.name]:
            call(cmd, shell=True)

    main()

    write_toc()

    if "post_cmd" in config and os.name in config["post_cmd"]:
        print("Running post-cmd")
        for cmd in config["post_cmd"][os.name]:
            call(cmd, shell=True)
