----------------------------------------------------------------------------------
-- Company: 
-- Engineer: 
-- 
-- Create Date: sept.2018 
-- Design Name: 
-- Module Name: uart_tx_ctrl - Behavioral
-- 
-- Dependencies: 
-- 
-- FSM using 3 processes for CERN course
-- 
----------------------------------------------------------------------------------


library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

use IEEE.NUMERIC_STD.ALL;


entity uart_tx_ctrl is
    Port ( tx_data_in : in STD_LOGIC_VECTOR (7 downto 0);
           tx_start : in STD_LOGIC;
           tx_baud_rate_enable : in STD_LOGIC;
           tx_serial_out : out STD_LOGIC;
           clock : in STD_LOGIC;
           reset : in STD_LOGIC);
end uart_tx_ctrl;

architecture Behavioral of uart_tx_ctrl is

signal count_value: integer range 0 to 7 := 0;
signal count_reset: std_logic := '0';
    -- signals associated with the one-shot
signal start_transmitting   : std_logic := '0';
signal reset_oneShot        : std_logic := '0';    
signal registered_data      : std_logic_vector(7 downto 0) := (others=>'0');
   -- state machine defintion
type valid_states is (IDLE, START, DATA, STOP);
signal state, next_state  : valid_states := IDLE;

begin

   process (clock)
   begin
      if clock='1' and clock'event then
         if reset='1' or count_reset = '1' then
            count_value <= 0;
         elsif tx_baud_rate_enable='1' then
            count_value <= count_value + 1;
         end if;
      end if;
   end process;

    -- one-shot to capture short duration events 
    -- in this case, the tx_start
    oneShot: process (clock)
    begin
        if rising_edge(clock) then		    -- active only on clock edge
            if (reset ='1') or 			    -- is external reset asserted?
                (reset_oneShot = '1') then	-- what about a reset from the FSM?
                start_transmitting <= '0';	-- don't tell the FSM to begin
            else				            -- reset not asserted
                if (tx_start = '1') then		-- has a start been requested?
                    start_transmitting <= '1';	-- let the FSM know
                    registered_data <= tx_data_in;	-- store the data while valid
                end if;		-- end of tx_start test				
            end if;			-- end of reset/non-reset activities
        end if;				-- end of synch events
    end process oneShot;

-- Finite State Machine for controlling serial tramsission
-- modeled using 3 process. Next_state combinational. Combinational outputs, sequential state transitions
   FSM_nextST_comb: process (state, start_transmitting, count_value) --controll transitions
   begin
      case state is
           when IDLE =>
               if (start_transmitting = '1') then   -- transition when request is made
                   next_state <= START;                 -- to the START state
               end if;
                           
           when START =>                                    
                  next_state <= DATA;             -- always happens
           when DATA =>               
               if (count_value = 7) then   -- if done
                   next_state <= STOP;
               end if;               
           when STOP =>                                                  
               -- transitions
               next_state <= IDLE;              -- always
       end case;   -- end of states         
   end process FSM_nextST_comb;

   FSM_tx_reg: process (clock)
   begin
        if rising_edge(clock) then      -- active only on clock edge
            if (reset = '1') then		-- is the external reset asserted?
                state <= IDLE;
            else						-- reset not asserted
                if (tx_baud_rate_enable = '1') then -- don't operate every clock, only @ baud
                   state <= next_state;
                end if;			-- end of baud enable test
            end if;				-- end of reset/non-reset activities
        end if;					-- end of synch activities
    end process FSM_tx_reg;    


    FSM_tx_out: process (state) -- actions
    begin
        case state is
            when IDLE =>
                tx_serial_out <= '1';	-- idle the RS-232 line
                reset_oneShot <= '0';	-- enable the one-shot to look for xmit requests
                count_reset   <= '1';	-- preload the counter and don't start counting
            when START =>
                tx_serial_out <= '0';		-- start bit is always 0
                count_reset <= '0';			-- let the counter count
            when DATA =>
                tx_serial_out <= registered_data(count_value);	-- select the proper bit
                if (count_value = 7) then
                    count_reset <= '1';
                end if;                
            when STOP =>
                tx_serial_out <= '1';		-- stop bit is always line idle
                reset_oneShot <= '1';		-- clear the 1shot for next event
        end case;	-- end of states			
    end process FSM_tx_out;    

end Behavioral;
